// EGE Preparation Website - Version 3 (HTML/CSS/JS) - Dashboard Design

const subjects = {
  math: {
    id: 'math',
    name: 'Математика',
    icon: 'calculator.svg',
    gradient: 'linear-gradient(135deg, #667EEA 0%, #764BA2 100%)',
    description: 'Профильная математика с углубленным изучением всех разделов',
    duration: '120 часов',
    price: 'от 15000₽',
    stats: { students: '1200+', success: '94%' }
  },
  russian: {
    id: 'russian',
    name: 'Русский язык',
    icon: 'book-open.svg',
    gradient: 'linear-gradient(135deg, #F093FB 0%, #F5576C 100%)',
    description: 'Подготовка к сочинению и тестовой части',
    duration: '100 часов',
    price: 'от 12000₽',
    stats: { students: '980+', success: '96%' }
  },
  physics: {
    id: 'physics',
    name: 'Физика',
    icon: 'zap.svg',
    gradient: 'linear-gradient(135deg, #4FACFE 0%, #00F2FE 100%)',
    description: 'Механика, термодинамика, оптика и квантовая физика',
    duration: '140 часов',
    price: 'от 18000₽',
    stats: { students: '750+', success: '92%' }
  },
  informatics: {
    id: 'informatics',
    name: 'Информатика',
    icon: 'code.svg',
    gradient: 'linear-gradient(135deg, #43E97B 0%, #38F9D7 100%)',
    description: 'Программирование, алгоритмы и структуры данных',
    duration: '130 часов',
    price: 'от 16000₽',
    stats: { students: '650+', success: '91%' }
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval
let selectedSubject = ''
let selectedDate = ''

function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadCourses()
  updateStats()
  
  setInterval(() => {
    loadCourses()
    updateStats()
  }, 500)
}

function loadCourses() {
  const saved = localStorage.getItem('egeCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    courses = [
      {
        id: 1,
        subject: 'Математика',
        instructor: 'Александр Петров',
        progress: 75,
        testResults: [
          { name: 'Пробный ЕГЭ №1', score: 82, maxScore: 100 },
          { name: 'Пробный ЕГЭ №2', score: 88, maxScore: 100 }
        ]
      },
      {
        id: 2,
        subject: 'Русский язык',
        instructor: 'Елена Соколова',
        progress: 60,
        testResults: [
          { name: 'Пробный ЕГЭ №1', score: 90, maxScore: 100 }
        ]
      },
      {
        id: 3,
        subject: 'Физика',
        instructor: 'Дмитрий Новиков',
        progress: 45,
        testResults: []
      }
    ]
    localStorage.setItem('egeCourses', JSON.stringify(courses))
  }
  
  renderCourses(courses)
}

function renderCourses(courses) {
  const container = document.getElementById('courses-container')
  if (!container) return
  
  if (courses.length === 0) {
    container.innerHTML = `
      <div class="empty-widget">
        <div class="empty-icon-modern">
          <img src="icons/book-open.svg" alt="BookOpen" width="48" height="48">
        </div>
        <h3 class="empty-title-modern">Начните обучение</h3>
        <p class="empty-text-modern">Запишитесь на первый курс подготовки к ЕГЭ</p>
      </div>
    `
    return
  }
  
  container.innerHTML = courses.map(course => {
    return `
      <div class="course-module">
        <div class="module-header">
          <div class="module-badge">${course.subject}</div>
          <div class="module-progress-badge">${course.progress}%</div>
        </div>
        <div class="module-body">
          <div class="module-instructor">
            <img src="icons/user.svg" alt="User" width="16" height="16">
            <span>${course.instructor}</span>
          </div>
          <div class="module-progress-track">
            <div class="module-progress-fill" style="width: ${course.progress}%"></div>
          </div>
        </div>
        ${course.testResults.length > 0 ? `
          <div class="module-tests">
            <div class="tests-header-modern">
              <img src="icons/bar-chart-3.svg" alt="BarChart" width="16" height="16">
              <span>Результаты</span>
            </div>
            <div class="tests-list-modern">
              ${course.testResults.map(test => `
                <div class="test-module">
                  <div class="test-name-modern">${test.name}</div>
                  <div class="test-score-modern">
                    <span class="score-highlight">${test.score}</span>
                    <span class="score-divider">/</span>
                    <span>${test.maxScore}</span>
                  </div>
                </div>
              `).join('')}
            </div>
          </div>
        ` : ''}
      </div>
    `
  }).join('')
}

function updateStats() {
  const saved = localStorage.getItem('egeCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  }
  
  const totalProgress = courses.length > 0
    ? Math.round(courses.reduce((sum, c) => sum + c.progress, 0) / courses.length)
    : 0
  
  const averageScore = courses.length > 0
    ? Math.round(
        courses
          .flatMap(c => c.testResults)
          .reduce((sum, t) => sum + (t.score / t.maxScore) * 100, 0) /
        courses.flatMap(c => c.testResults).length
      )
    : 0
  
  const totalTests = courses.flatMap(c => c.testResults).length
  
  const totalProgressEl = document.getElementById('total-progress')
  const totalCoursesEl = document.getElementById('total-courses')
  const averageScoreEl = document.getElementById('average-score')
  const totalTestsEl = document.getElementById('total-tests')
  const coursesCountEl = document.getElementById('courses-count')
  const progressRing = document.getElementById('progress-ring')
  
  if (totalProgressEl) totalProgressEl.textContent = totalProgress + '%'
  if (totalCoursesEl) totalCoursesEl.textContent = courses.length
  if (averageScoreEl) averageScoreEl.textContent = averageScore || 0
  if (totalTestsEl) totalTestsEl.textContent = totalTests
  if (coursesCountEl) coursesCountEl.textContent = courses.length + ' курсов'
  
  if (progressRing) {
    const circumference = 2 * Math.PI * 54
    const offset = circumference - (totalProgress / 100) * circumference
    progressRing.style.strokeDasharray = `${circumference} ${circumference}`
    progressRing.style.strokeDashoffset = offset
    progressRing.style.stroke = 'url(#progressGradient)'
  }
}

function initSlider() {
  slides = document.querySelectorAll('.slide')
  dots = document.querySelectorAll('.dot')
  
  if (slides.length === 0) return
  
  updateSlider()
  
  slideInterval = setInterval(() => {
    currentSlide = (currentSlide + 1) % slides.length
    updateSlider()
  }, 3000)
}

function updateSlider() {
  slides.forEach((slide, index) => {
    if (index === currentSlide) {
      slide.classList.add('active')
    } else {
      slide.classList.remove('active')
    }
  })
  
  dots.forEach((dot, index) => {
    if (index === currentSlide) {
      dot.classList.add('active')
    } else {
      dot.classList.remove('active')
    }
  })
}

function goToPrevious() {
  if (slides.length === 0) return
  currentSlide = (currentSlide - 1 + slides.length) % slides.length
  updateSlider()
  if (slideInterval) clearInterval(slideInterval)
  slideInterval = setInterval(() => {
    currentSlide = (currentSlide + 1) % slides.length
    updateSlider()
  }, 3000)
}

function goToNext() {
  if (slides.length === 0) return
  currentSlide = (currentSlide + 1) % slides.length
  updateSlider()
  if (slideInterval) clearInterval(slideInterval)
  slideInterval = setInterval(() => {
    currentSlide = (currentSlide + 1) % slides.length
    updateSlider()
  }, 3000)
}

function goToSlide(index) {
  if (slides.length === 0) return
  currentSlide = index
  updateSlider()
  if (slideInterval) clearInterval(slideInterval)
  slideInterval = setInterval(() => {
    currentSlide = (currentSlide + 1) % slides.length
    updateSlider()
  }, 3000)
}

function initEnrollPage() {
  renderSubjects()
  
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
    
    dateInput.addEventListener('change', (e) => {
      selectedDate = e.target.value
      updateEnrollProgress()
      const submitBtn = document.getElementById('submit-btn')
      if (submitBtn) submitBtn.disabled = !selectedSubject || !selectedDate
    })
  }
  
  const form = document.getElementById('enroll-form')
  if (form) {
    form.addEventListener('submit', handleSubmit)
  }
}

function renderSubjects() {
  const container = document.getElementById('subjects-container')
  if (!container) return
  
  container.innerHTML = Object.values(subjects).map(subject => {
    return `
      <button 
        type="button" 
        class="subject-showcase-card" 
        data-subject="${subject.id}"
        style="border-color: #E2E8F0;"
      >
        <div class="showcase-icon-wrapper">
          <div class="showcase-icon" style="background: ${subject.gradient}15; border-color: transparent;">
            <img src="icons/${subject.icon}" alt="${subject.name}" width="48" height="48">
          </div>
        </div>
        <div class="showcase-content">
          <h3 class="showcase-name" style="color: #1E293B;">${subject.name}</h3>
          <p class="showcase-description" style="color: #64748B;">${subject.description}</p>
          <div class="showcase-stats">
            <div class="stat-item-modern">
              <img src="icons/users.svg" alt="Users" width="16" height="16">
              <span>${subject.stats.students}</span>
            </div>
            <div class="stat-item-modern">
              <img src="icons/trending-up.svg" alt="TrendingUp" width="16" height="16">
              <span>${subject.stats.success}</span>
            </div>
          </div>
          <div class="showcase-meta">
            <div class="meta-item-modern">
              <img src="icons/clock.svg" alt="Clock" width="16" height="16">
              <span>${subject.duration}</span>
            </div>
            <div class="meta-item-modern price">
              <span>${subject.price}</span>
            </div>
          </div>
        </div>
      </button>
    `
  }).join('')
  
  const subjectCards = document.querySelectorAll('.subject-showcase-card')
  subjectCards.forEach(card => {
    card.addEventListener('click', () => {
      const subjectId = card.dataset.subject
      selectedSubject = subjectId
      
      subjectCards.forEach(c => {
        c.classList.remove('selected')
        const subject = subjects[c.dataset.subject]
        
        if (c === card) {
          c.style.background = subject.gradient
          c.style.borderColor = 'transparent'
          c.style.color = 'white'
          const icon = c.querySelector('.showcase-icon')
          const name = c.querySelector('.showcase-name')
          const desc = c.querySelector('.showcase-description')
          const stats = c.querySelectorAll('.stat-item-modern')
          const meta = c.querySelectorAll('.meta-item-modern')
          
          if (icon) {
            icon.style.background = 'rgba(255, 255, 255, 0.25)'
            icon.style.borderColor = 'rgba(255, 255, 255, 0.4)'
            icon.querySelector('img').style.filter = 'brightness(0) invert(1)'
          }
          if (name) name.style.color = 'white'
          if (desc) desc.style.color = 'rgba(255, 255, 255, 0.9)'
          stats.forEach(s => {
            s.style.color = 'rgba(255, 255, 255, 0.9)'
            s.querySelector('img').style.filter = 'brightness(0) invert(1)'
          })
          meta.forEach(m => {
            m.style.color = 'rgba(255, 255, 255, 0.9)'
            const img = m.querySelector('img')
            if (img) img.style.filter = 'brightness(0) invert(1)'
          })
          
          c.innerHTML += `
            <div class="showcase-check">
              <img src="icons/check-circle.svg" alt="Check" width="32" height="32">
            </div>
          `
        } else {
          c.style.background = 'white'
          c.style.borderColor = '#E2E8F0'
          c.style.color = '#1E293B'
          const icon = c.querySelector('.showcase-icon')
          const name = c.querySelector('.showcase-name')
          const desc = c.querySelector('.showcase-description')
          const stats = c.querySelectorAll('.stat-item-modern')
          const meta = c.querySelectorAll('.meta-item-modern')
          
          if (icon) {
            icon.style.background = `${subject.gradient}15`
            icon.style.borderColor = 'transparent'
            icon.querySelector('img').style.filter = 'none'
          }
          if (name) name.style.color = '#1E293B'
          if (desc) desc.style.color = '#64748B'
          stats.forEach(s => {
            s.style.color = '#64748B'
            s.querySelector('img').style.filter = 'none'
          })
          meta.forEach(m => {
            m.style.color = '#64748B'
            const img = m.querySelector('img')
            if (img) img.style.filter = 'none'
          })
          
          const check = c.querySelector('.showcase-check')
          if (check) check.remove()
        }
      })
      
      updateEnrollProgress()
      showDateSection()
    })
  })
}

function updateEnrollProgress() {
  const step1Indicator = document.getElementById('step-1-indicator')
  const step2Indicator = document.getElementById('step-2-indicator')
  const progress1 = document.getElementById('progress-1')
  
  if (selectedSubject) {
    if (step1Indicator) {
      step1Indicator.classList.add('completed')
      step1Indicator.innerHTML = '<img src="icons/check-circle.svg" alt="Check" width="20" height="20">'
    }
    if (progress1) progress1.style.width = '100%'
    if (step2Indicator) step2Indicator.classList.add('active')
  }
  
  if (selectedDate && selectedSubject) {
    if (step2Indicator) {
      step2Indicator.classList.add('completed')
      step2Indicator.innerHTML = '<img src="icons/check-circle.svg" alt="Check" width="20" height="20">'
    }
  }
}

function showDateSection() {
  const dateSection = document.getElementById('date-section')
  const submitBtn = document.getElementById('submit-btn')
  
  if (selectedSubject) {
    if (dateSection) dateSection.style.display = 'block'
  }
  
  if (submitBtn) submitBtn.disabled = !selectedSubject || !selectedDate
}

function handleSubmit(e) {
  e.preventDefault()
  if (!selectedSubject || !selectedDate) return
  
  const saved = localStorage.getItem('egeCourses')
  const courses = saved ? JSON.parse(saved) : []
  
  const subjectData = subjects[selectedSubject]
  
  const newCourse = {
    id: Date.now(),
    subject: subjectData.name,
    instructor: 'Преподаватель',
    progress: 0,
    testResults: []
  }
  
  courses.push(newCourse)
  localStorage.setItem('egeCourses', JSON.stringify(courses))
  
  const successMessage = document.getElementById('success-message')
  const submitBtn = document.getElementById('submit-btn')
  
  if (successMessage) successMessage.style.display = 'flex'
  if (submitBtn) submitBtn.style.display = 'none'
  
  setTimeout(() => {
    window.location.href = 'index.html'
  }, 2000)
}

document.addEventListener('DOMContentLoaded', initPage)

